import { setRequestLocale } from "next-intl/server";
import { Navbar } from "@/components/navbar";
import { Footer } from "@/modules/landing/components";
import { getBlogs, getRecentBlogs, getTags } from "@/core/lib/api";
import { BlogCard, BlogSidebar, BlogHeader, BlogPagination } from "@/modules/blog/components";

type Props = {
  params: Promise<{ locale: string }>;
  searchParams: Promise<{ page?: string; tag?: string }>;
};

export default async function BlogListingPage({ params, searchParams }: Props) {
  const { locale } = await params;
  const { page: pageStr, tag } = await searchParams;
  setRequestLocale(locale);

  const currentPage = Math.max(1, parseInt(pageStr || "1", 10) || 1);

  const [blogsResult, tags, recentPosts] = await Promise.all([
    getBlogs({ page: currentPage, pageSize: 9, tag }),
    getTags(),
    getRecentBlogs(5),
  ]);

  const { data: blogs, meta } = blogsResult;
  const pagination = meta?.pagination;

  return (
    <div className="min-h-screen bg-slate-950">
      <Navbar />

      <BlogHeader
        title="Blog"
        subtitle="Insights, tutorials, and updates from our team"
      />

      <div className="container py-12 md:py-16">
        <div className="grid gap-8 lg:grid-cols-[1fr_320px]">
          {/* Main Content */}
          <main>
            {blogs.length > 0 ? (
              <>
                <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-2">
                  {blogs.map((blog, index) => (
                    <BlogCard
                      key={blog.documentId}
                      blog={blog}
                      index={index}
                      featured={index === 0 && currentPage === 1}
                    />
                  ))}
                </div>

                {pagination && (
                  <BlogPagination
                    currentPage={pagination.page}
                    pageCount={pagination.pageCount}
                    tag={tag}
                  />
                )}
              </>
            ) : (
              <div className="flex flex-col items-center justify-center py-24 text-center">
                <div className="mb-4 rounded-full bg-slate-800/50 p-6">
                  <svg
                    className="h-12 w-12 text-slate-600"
                    fill="none"
                    viewBox="0 0 24 24"
                    stroke="currentColor"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={1.5}
                      d="M19 20H5a2 2 0 01-2-2V6a2 2 0 012-2h10a2 2 0 012 2v1m2 13a2 2 0 01-2-2V7m2 13a2 2 0 002-2V9a2 2 0 00-2-2h-2m-4-3H9M7 16h6M7 8h6v4H7V8z"
                    />
                  </svg>
                </div>
                <h3 className="text-lg font-semibold text-slate-300">
                  No articles found
                </h3>
                <p className="mt-2 text-sm text-slate-500">
                  {tag
                    ? `No articles with the tag "${tag}" yet. Try a different tag.`
                    : "Check back soon for new content."}
                </p>
              </div>
            )}
          </main>

          {/* Sidebar */}
          <BlogSidebar
            tags={tags}
            recentPosts={recentPosts}
            activeTag={tag}
          />
        </div>
      </div>

      <Footer />
    </div>
  );
}
